<?php
/*
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */
namespace Xtwo\Automationshell\Console;

ini_set("max_execution_time", 0);

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Exception\FileNotFoundException;

class AttributeimagedefaultCommand extends Command
{
    /**
     * Target store parameter code
     */
    const PARAM_TARGET_STORE = 'store';

    /**
     * SKU parameter code
     */
    const PARAM_SKU = 'sku';

    /**
     * ALL products parameter code
     */
    const PARAM_ALL = 'all';

    const PARAM_SORT_ONLY = 'sort-order-only';

    /**
     * File to log messages to
     */
    const LOGFILE = 'js_attributeimagedefault.log';

    /**
     * Chunk size to update
     */
    const CHUNK_SIZE = 2000;

    /**
     * Max. limit products for varnish ban
     */
    const VARNISH_BAN_MAX_LIMIT = 200;

    /**
     * List of required parameters
     *
     * @var array
     */
    protected $requiredParameters = [

    ];

    /**
     * List of attributes to reset to default store
     *
     * @var array
     */
    protected $attributeList = [
        'thumbnail',
        'small_image',
        'image'
    ];

    /**
     * Map of attributes and their tables
     *
     * @var array
     */
    protected $attributeMap = [];

    /**
     * Product action model
     *
     * @var \Magento\Catalog\Model\Product\Action
     */
    protected $productAction;

    /**
     * Product model
     *
     * @var \Magento\Catalog\Model\Product
     */
    protected $productModel;

    /**
     * Varnish ban model
     *
     * @var Nexcessnet_Turpentine_Model_Observer_Ban
     */
    protected $varnish;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    protected $dataObjectFactory;

    /**
     * @var 
     */
 
    public function __construct(
        \Psr\Log\LoggerInterface $logger,
        \Magento\Framework\DataObjectFactory $dataObjectFactory,
        \Magento\Catalog\Model\ResourceModel\Product\Action $productAction,
        \Magento\Catalog\Model\Product $product,
        \Magento\Store\Model\Store $store,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollection,
        \Magento\Framework\App\ResourceConnection $resourceConnection
        ) {
        $this->resourceConnection = $resourceConnection;
        $this->connection = $this->resourceConnection->getConnection();
        $this->store = $store;
        $this->storeManager = $storeManager;
        $this->productModel = $product;
        $this->productAction = $productAction;

        $this->dataObjectFactory = $dataObjectFactory;
       
        $this->logger = $logger;
        $this->productCollection = $productCollection;
   
        parent::__construct();
    }
    /**
     * Function to log error and message
     * @param string $msg
     * @return Mage_Log_Model_Log
     * @author Metawolf
     */
    protected function log($msg)
    {
        $this->logger->log(\Monolog\Logger::INFO, $msg);
    }

    /**
     * Function to echo error and log to file
     * @param $msg= message,$isIntro= intro,$skipLineBreak= adding line break
     * @return message
     * @author Metawolf
     *
     * @param string $msg
     * @param bool   $isIntro
     * @param bool   $skipLineBreak
     */
    protected function echoAndLog($msg, $isIntro = false, $skipLineBreak = false)
    {
        $echo = ($isIntro) ? $msg : '   ' . $msg;
        $lb = $skipLineBreak ? '' : "\n";
        echo $echo . $lb;
        $this->log($echo);
    }

    /**
     * Function to Throw exception
     * @param $message
     * @throws \Exception
     */
    protected function throwException($message)
    {
        throw new \Exception($message);
    }

    /**
     * Init function
     */
    protected function inits()
    {
       $this->checkRequiredFields();
       $this->prepareAttributes();
    }

    protected function configure()
    {
      
        $this->setName('Xtwo:AttributeimagedefaultCommand')
            ->setDescription('Resets the Default Image Selection (on store-view level) back to Default (Backend: "Use Default Value"). Usable to do that for
            a specific product, a specific store, or ALL products on ALL stores. ')         
            ->addOption('store', "store", InputOption::VALUE_OPTIONAL, "--stores xtwostore_de,xtwostore_ch")
            ->addOption('sku', "s", InputOption::VALUE_OPTIONAL, "--sku 34840000,34841000")
            ->addOption('all',"a",InputOption::VALUE_OPTIONAL, "--all To ensure that a reset to default is really desired for ALL stores")
            ->addOption('sort-order-only',"so",InputOption::VALUE_OPTIONAL, "--sort-order-only If specified, it only adjusts the sort order of the existing store-releated images 
            to the order specified in DEFAULT store. With this option the --store option is ignored.");
            
            parent::configure();
    }

    /**
     * Check required fields existence
     */
    protected function checkRequiredFields()
    {
        //check for required parameters
        foreach ($this->requiredParameters as $parameter) {

            if (!$this->input->getOption($parameter)) {
              $this->throwException(sprintf('Required parameter %s missing', $parameter));
            }
        }

        //if --sku parameters or --all not provided
        if (!$this->input->getOption(self::PARAM_SKU) && !$this->input->getOption(self::PARAM_TARGET_STORE) && !$this->input->getOption(self::PARAM_ALL)) {
            $this->throwException(sprintf('Required parameter "%s" missing to ensure reset to ALL', self::PARAM_ALL));
        }
    }

    /**
     * Function to get sku list from provided shell parameter
     * @param string $skuParamValue
     * @return array
     */
    protected function getSkuList($skuParamValue)
    {
        $skuList = [];

        if ($skuParamValue) {
            $skuList = explode(',', trim($skuParamValue));
        }

        return $skuList;
    }

    /**
     * Function to get store ID by store code
     * @param $code
     * @return $store id int|null
     * @throws \Exception
     */
    protected function getStoreIdByCode($code)
    {
        $storeId = null;

        /** @var \Magento\Store\Model\Store $store */
        
        // $store = Mage::getModel('core/store')->load($code, 'code');
         $store = $this->store->load($code, 'code');

        if ($store && $store->getId()) {
            $storeId = $store->getId();
        }

        if ($storeId === "0") {
            // hint a big mistake
            $this->throwException(sprintf("Your're changing default values!", $code));
        }
        if (!$storeId) {
            $this->throwException(sprintf('Store with code %s does not exist', $code));
        }

        return $storeId;
    }

    /**
     * Get product collection
     *
     * @param array $skuList
     *
     * @return \Magento\Catalog\Model\ResourceModel\Product\Collection
     */
    protected function getProducts($skuList)
    {
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $collection */
       // $collection = Mage::getResourceModel('catalog/product_collection');

        $collection = $this->productCollection->create();

        //if sku list provided, as as filter
        if (!empty($skuList)) {
            $collection->addAttributeToFilter('sku', array('in' => $skuList));
        }

        $collection->setPageSize(self::CHUNK_SIZE);
        $collection->setCurPage(1);
        $collection->setOrder('entity_id', 'asc');

        return $collection;
    }

    /**
     * Prepare product table and id and store them
     *
     * @return array
     */
    protected function prepareAttributes()
    {
        if (!$this->attributeMap) {
            $attributeMap = [];
            foreach ($this->attributeList as $attributeCode) {
                $attribute = $this->productModel->getResource()->getAttribute($attributeCode);
                if ($attribute) {
                    $attributeMap[$attribute->getBackend()->getTable()][] = $attribute->getId();
                }
            }
            $this->attributeMap = $attributeMap;
        }

        return $this->attributeMap;
    }

    /**
     * Reset attributes to admin values for provided product ids and store id
     * @param $sourceProductIds
     * @param $storeId int target store ID
     * @return int the number of real updated products
     */
    protected function resetImagesToDefault($sourceProductIds=NULL, $storeId=NULL)
    {
        $realUpdates = null;
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
       // $adapter = $this->productModel->getResource()->getWriteConnection();
        $adapter = $this->connection;
        foreach ($this->attributeMap as $table => $attributeIds) {
            $ids = implode(',', $attributeIds);
            

            if(is_array($sourceProductIds)){
                $entityIds = count($sourceProductIds) ? implode(',', $sourceProductIds) : '';
            }  else {
                $entityIds = ''; 
            }  
            $where = sprintf("attribute_id in (%s) AND %s %s",
                $ids, $storeId ? "store_id = '{$storeId}'" : 'store_id > 0',
                !empty($entityIds) ? "AND entity_id in ({$entityIds})" : '');
          //  $deletionCount = $adapter->delete($table, $where);

         $sql = <<<EOT
          DELETE FROM {$table}
          WHERE attribute_id IN (SELECT attribute_id FROM eav_attribute where entity_type_id = 4 and attribute_id in ($ids)) and $where;
          EOT;

          $deletionCount = $this->resourceConnection->getConnection()->query($sql);               
          $realUpdates += 1;
        }

        return $realUpdates;
    }


    protected function adjustImagePositionToDefault($sourceProductIds) {
        $totalImages = 0;
        $totalProducts = 0;
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
       // $adapter = $this->productModel->getResource()->getWriteConnection();

        $adapter = $this->connection;

        $attribute = $this->productModel->getResource()->getAttribute('media_gallery');
        foreach ($sourceProductIds as $productId) {
            $totalProducts++;
            $query = "SELECT value_id FROM catalog_product_entity_media_gallery WHERE attribute_id = {$attribute->getId()} AND entity_id = {$productId}";
            $valueIds = $adapter->query($query);
            foreach ($valueIds as $data) {
                $defaultPosition = null;
                $query = "SELECT store_id, position, value_id FROM catalog_product_entity_media_gallery_value WHERE value_id = {$data['value_id']} ORDER BY store_id ASC";
                $entries = $adapter->fetchAll($query);
                foreach ($entries as $entry) {
                    if ($entry['store_id'] == 0) {
                        $defaultPosition = $entry['position'];
                        continue;
                    }
                    if (!is_null($defaultPosition) && $entry['store_id']>0 && $entry['position'] != $defaultPosition) {
                        $update = "UPDATE catalog_product_entity_media_gallery_value SET position = {$defaultPosition} WHERE value_id = {$entry['value_id']} AND store_id = {$entry['store_id']}";
                        $adapter->query($update);
                        $totalImages++;
                    }
                }
            }
            if ($totalProducts % 5000 == 0) {
                $this->echoAndLog(sprintf('Processed bunch of 5000.... to %s. Images adjusted to default in total: %s',
                    $totalProducts, $totalImages));
            }
        }
        return $totalImages;
    }

    /**
     * Ban varnish URLs for updated products
     * @param $storeId
     * @param $productIds
     * @throws \Magento\Framework\Exception\LocalizedException
     */
   
     /*
    protected function banVarnishUrls($productIds, $storeId)
    {
        if (is_null($storeId)) {
            $this->echoAndLog(sprintf(' ... skipped as no storeID given to ban varnish.'), false, true);
            return;
        }
        if (count($productIds) > self::VARNISH_BAN_MAX_LIMIT) {
            $this->echoAndLog(sprintf(' ... skipped as limit reached to ban varnish.'), false, true);
            return;
        }
        $this->echoAndLog(sprintf('Clearing varnish cache for %s products', count($productIds)), false, true);
        foreach ($productIds as $productId) {
            $product = $this->productModel->reset()->setStoreId($storeId)->load($productId);
            $event = $this->dataObjectFactory->create();
            $this->varnish->banProductPageCache($event->setProduct($product));
        }
        $this->echoAndLog(sprintf(' ... done'), false, true);
    } 
    */

    protected function getStores() {
        $storeCodes = $this->input->getOption(self::PARAM_TARGET_STORE);
        if ($storeCodes) {
            $storeCodes = explode(',', $storeCodes);
            $stores = array();
            foreach ($storeCodes as $storeCode) {
                $stores[] = $this->storeManager->getStore($storeCode);
            }
        } else {
            $stores = $this->storeManager->getStores();
        }
        return $stores;
    }
   
    /**
     * Run script
     */
    protected function execute(InputInterface $input, OutputInterface $output)  
    {
         $this->input = $input;
         $this->inits();
        //collect parameters
       $targetCode = $this->input->getOption(self::PARAM_TARGET_STORE);
       
        $skuParam = $this->input->getOption(self::PARAM_SKU);
       
        $skuList = $this->input->getOption(self::PARAM_SKU);

        $stores = $this->getStores();
       
        try {
            //init models, check required fields
           $this->inits();
          
            $productIds = [];

            //get store ids and product collections
            $targetStoreId = $this->getStoreIdByCode($targetCode);
            if (isset($this->input->getOption[self::PARAM_SORT_ONLY])) $targetStoreId = NULL;
            $typeOfProcess = isset($this->input->getOption[self::PARAM_SORT_ONLY]) ?
                'Sort order adjustment for gallery images' : 'Reset to default images';

            if (empty($skuList) && empty($targetStoreId)) {
                $this->echoAndLog(sprintf('%s for ALL products in 5 sec!', $typeOfProcess));
                sleep(5);
            } elseif (empty($skuList)) {
                $this->echoAndLog(sprintf('%s for all products of store %s in 5 sec!', $typeOfProcess, $targetCode));
                sleep(5);
            } else {
                $this->echoAndLog(sprintf('%s for %s', $typeOfProcess, $targetCode ? 'store '.$targetCode : 'all stores'));
            }

            if (!empty($skuList)) {
                //get source products and values
                $sourceCollection = $this->getProducts($skuList);
                $pages = $sourceCollection->getLastPageNumber();
                $currentPage = 1;

                if (!$sourceCollection->getSize()) {
                    $this->throwException(sprintf('No products found for store %s', $targetCode));
                }

                do {
                    //get all ids with offset of 0 first, and chunk-size + pagenum on other passes
                    $sourceProductIds = $sourceCollection->getAllIds(
                        self::CHUNK_SIZE,
                        self::CHUNK_SIZE * ($currentPage - 1)
                    );

                    if ($sourceCollection->getSize()) {
                        //chunkSize
                        if ($sourceCollection->getSize() < $sourceCollection->getPageSize()) {
                            $chunkSize = $sourceCollection->getSize();
                        } else {
                            $chunkSize = $sourceCollection->getPageSize();
                        }

                        $firstProductId = reset($sourceProductIds);
                        $this->echoAndLog(
                            sprintf('Processing chunk of %s product(s), start@%s', $chunkSize, $firstProductId),
                            false,
                            true
                        );

                        $shortonly = $this->input->getOption(self::PARAM_SORT_ONLY);

                        if (isset($shortonly)) {
                           // die();
                            $realUpdates = $this->adjustImagePositionToDefault($sourceProductIds);

                        } else {
                            //set target values and get affected products
                            $realUpdates = $this->resetImagesToDefault($sourceProductIds, $targetStoreId);

                        }
                        $productIds = array_merge($productIds, $sourceProductIds);

                        $this->echoAndLog(sprintf(' ... done (on real %s images / %s product(s))',
                                            $realUpdates, count($productIds)), false);
                    }
                    $currentPage++;
                    $sourceCollection->clear();
                } while ($currentPage <= $pages);

                //clear varnish
              //  $this->banVarnishUrls($productIds, $targetStoreId);

            } else {
                if (isset($this->_args[self::PARAM_SORT_ONLY])) {

                    $collection = Mage::getModel('catalog/product');
                    $collection->getSelect()->reset(\Zend_Db_Select::COLUMNS);
                    $collection->getSelect()->columns('entity_id');
                    $productIds = $collection->getAllIds();

                    $realUpdates = $this->adjustImagePositionToDefault($productIds);

                } else {
                    $realUpdates = $this->resetImagesToDefault(NULL, $targetStoreId);
                }
                if (!empty($targetStoreId)) {
                    $this->echoAndLog(sprintf(' ... done for store %s on %s images', $targetCode, $realUpdates), false);
                } else {
                    $this->echoAndLog(sprintf(' ... done for all stores (on total %s images)', $realUpdates), false);
                }
            }

        } catch (Exception $e) {
            $this->echoAndLog(sprintf('Stopped processing, Reason Exception: %s', $e->getMessage()));
            $this->echoAndLog("\n\n" . $this->usageHelp());
        }
    }

    /**
     * Retrieve Usage Help Message
     *
     */
    public function usageHelp()
    {
        return <<<USAGE
php bin/magento Xtwo:AttributeimagedefaultCommand --store xtwostorefr --sku Testtoday
       
Resets the Default Image Selection (on store-view level) back to Default (Backend: "Use Default Value"). Usable to do that for
a specific product, a specific store, or ALL products on ALL stores. 
        
Usage:   --[options]

  --store <store_code>    Store to reset values to default 
  
  --sku <sku_list>      List of SKUs to reset images to
  
  --all                 To ensure that a reset to default is really desired for ALL stores

  --sort-order-only     If specified, it only adjusts the sort order of the existing store-releated images 
                        to the order specified in DEFAULT store. With this option the --store option is ignored.

  help                  This help
  
  <store_code>  Store code, e.g. xtwostore_de, xtwostore_com etc. Optional.
  <sku_list>    List of product SKUs. Optional.

USAGE;
    }
}

