<?php

namespace Xtwo\AutoFillSearch\Model;

use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Escaper;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Store\Model\StoreManagerInterface;

class SearchSuggestion implements \Xtwo\AutoFillSearch\Api\SearchSuggestionInterface
{
    protected $productCollectionFactory;
    protected $categoryCollectionFactory;
    protected $scopeConfig;
    protected $escaper;
    protected $customApisHelper;
    protected $visibility;
    protected $storeManager;
    protected $taxCalculation;

    public function __construct(
        ProductCollectionFactory $productCollectionFactory,
        CategoryCollectionFactory $categoryCollectionFactory,
        ScopeConfigInterface $scopeConfig,
        Escaper $escaper,
        CustomApisHelper $customApisHelper,
        StoreManagerInterface $storeManager,
        \Magento\Tax\Model\Calculation $taxCalculation,
        Visibility $visibility
    ) {
        $this->productCollectionFactory = $productCollectionFactory;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->scopeConfig = $scopeConfig;
        $this->escaper = $escaper;
        $this->customApisHelper = $customApisHelper;
        $this->storeManager = $storeManager;
        $this->taxCalculation = $taxCalculation;
        $this->visibility = $visibility;
    }

    /**
     * @inheritDoc
     */
    public function getSearchSuggestions($searchkey)
    {
        if (strlen($searchkey) < 3) {
            return [
                'error' => true,
                'message' => 'Search key must have at least 3 characters.',
            ];
        }

        $isEnabled = $this->scopeConfig->getValue('searchautocomplete/popular/enabled', ScopeInterface::SCOPE_WEBSITE);
        $limit = $this->scopeConfig->getValue('searchautocomplete/popular/limit', ScopeInterface::SCOPE_STORE);

        if ($isEnabled != 1) {
            return [];
        }

        $suggestedCategoryCollection = $this->getSuggestedCategoryCollection($searchkey, $limit);
        $categoryIds = $suggestedCategoryCollection->getAllIds();
        
        $suggestedProductCollection = $this->getSuggestedProductCollection($searchkey, $limit, $categoryIds);

        $searchData = $this->prepareSuggestionData($suggestedProductCollection, 'Products');
        $categoryData = [
            'label' => 'Categories',
            'count' => $suggestedCategoryCollection->getSize(),
            'items' => $this->customApisHelper->getMainParentCategoriesNew($suggestedCategoryCollection),
        ];

        return [
            'search_data' => $searchData,
            'category_data' => $categoryData,
        ];
    }

    protected function getSuggestedProductCollection($searchkey, $limit, $categoryIds)
    {
        $storeId = $this->storeManager->getStore()->getId();
        $productCollection = $this->productCollectionFactory->create();
        $productCollection->addStoreFilter($storeId);
        $productCollection->addAttributeToSelect(['name', 'sku', 'thumbnail', 'url_key', 'price', 'msrp','tax_class_id']);
        $productCollection->addAttributeToFilter('visibility', 
            ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_SEARCH]]
        );
        $productCollection->addAttributeToFilter('status', 
            \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED
        );
        $productCollection->addAttributeToFilter('type_id', ['neq' => 'configurable']);
        $exactMatchCondition = [['attribute' => 'sku', 'eq' => $searchkey], ['attribute' => 'name', 'eq' => $searchkey]];
        
        if (strlen($searchkey) >= 3) {
            $productCollection->addAttributeToFilter([
                ['attribute' => 'sku', 'like' => '%' . $searchkey . '%'],
                ['attribute' => 'name', 'like' => '%' . $searchkey . '%'],
            ]);
        } else {
            $productCollection->addAttributeToFilter($exactMatchCondition, null, 'left');
        }
        if (!empty($categoryIds)) {
            $productCollection->addCategoriesFilter(['in' => $categoryIds]);
        }
        $productCollection->setPageSize($limit);
        return $productCollection;
    }

    protected function getSuggestedCategoryCollection($searchkey, $limit)
    {
        $categoryCollection = $this->categoryCollectionFactory->create();
        $categoryCollection->addAttributeToFilter('is_active', ['eq' => 1])
            ->addAttributeToSelect(['name','url_key'])
            ->addAttributeToFilter('name', ['like' => '%' . $searchkey . '%'])
            ->setPageSize($limit);

        return $categoryCollection;
    }

    protected function prepareSuggestionData($collection, $label, $isProduct = true)
    {
        $suggestionData = [
            'label' => $label,
            'count' => $collection->getSize(),
            'items' => [],
        ];

        foreach ($collection as $item) {
            $itemArray = [
                'name' => $item->getName(),
            ];

            if ($isProduct) {
                $taxClassId = $item->getTaxClassId();
                $store = $this->storeManager->getStore();
                $request = $this->taxCalculation->getRateRequest(null, null, null, $store);
                $taxRate = $this->taxCalculation->getRate($request->setProductClassId($taxClassId));

                $price = $item->getPrice();
                $basePrice = $price / 1.19;
                $priceWithTax = $basePrice * (1 + $taxRate / 100);
                
                $msrpPrice = $item->getMsrp();
                $msrpBasePrice = $msrpPrice / 1.19;
                $msrpPriceWithTax = $msrpBasePrice * (1 + $taxRate / 100);

                $itemArray += [
                    'sku' => $item->getSku(),
                    'image' => $item->getData('thumbnail'),
                    'product_url_key' => $item->getData('url_key'),
                    'price' => (string)$priceWithTax,
                    'msrp' => (string)$msrpPriceWithTax,
                ];
            } else {
                $itemArray += [
                    'category_url_key' => $item->getUrlKey(),
                ];
            }

            $suggestionData['items'][] = $itemArray;
        }

        return $suggestionData;
    }
}