<?php

namespace Xtwo\AutoFillSearch\Model;

use Xtwo\AutoFillSearch\Api\SearchResultInterface;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Magento\Wishlist\Model\WishlistFactory;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Xtwo\AttributeOption\Helper\Image as ImageHelper;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Store\Model\StoreManagerInterface;

class SearchResult implements SearchResultInterface
{
    protected $productCollectionFactory;
    protected $stockRegistry;
    protected $wishlistFactory;
    protected $categoryCollectionFactory;
    protected $imageHelper;
    protected $customApisHelper;
	protected $storeManager;
    protected $visibility;

    public function __construct(
        ProductCollectionFactory $productCollectionFactory,
        StockRegistryInterface $stockRegistry,
        WishlistFactory $wishlistFactory,
        CategoryCollectionFactory $categoryCollectionFactory,
        ImageHelper $imageHelper,
        CustomApisHelper $customApisHelper,
		StoreManagerInterface $storeManager,
		\Magento\Tax\Model\Calculation $taxCalculation,
        Visibility $visibility
    ) {
        $this->productCollectionFactory = $productCollectionFactory;
        $this->stockRegistry = $stockRegistry;
        $this->wishlistFactory = $wishlistFactory;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->imageHelper = $imageHelper;
        $this->customApisHelper = $customApisHelper;
		$this->storeManager = $storeManager;
		$this->taxCalculation = $taxCalculation;
        $this->visibility = $visibility;
    }

    public function getSearchResult($searchkey, $currentPage = null, $pagesize = null, $customerId = null, $attrCode= null, $attrValue= null, $attrprice= null, $lteq= null, $gteq= null, $sortAttribute= null, $sortDirection= null)
    {
        if (strlen($searchkey) < 3) {
            return [
                'error' => true,
                'message' => 'Search key must have at least 3 characters.',
            ];
        }

        if (!empty($attrCode)) {
            $cusRequest = array_combine($attrCode, $attrValue);
        }

        $result = [
            'products' => [],
            'categories' => [],
            'search_criteria' => [
                'page_size' => $pagesize,
                'current_page' => $currentPage,
            ],
            'total_count_products' => 0,
            'total_count_categories' => 0,
        ];

        try {
			$storeId = $this->storeManager->getStore()->getId();
            $productCollection = $this->productCollectionFactory->create();
            $productCollection->addAttributeToSelect('*');
			$productCollection->addStoreFilter($storeId);
            $productCollection->addAttributeToFilter(
                'visibility', 
                ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_SEARCH]]
            );
            $productCollection->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);
            $productCollection->addAttributeToFilter(
                [
                    ['attribute' => 'name', 'like' => '%' . $searchkey . '%'],
                    ['attribute' => 'sku', 'like' => '%' . $searchkey . '%']   
                ],
                null,
                'LEFT'
            );
            $productCollection->addAttributeToFilter('type_id', ['neq' => 'configurable']);
            $productCollection->setCurPage($currentPage);
            $productCollection->setPageSize($pagesize);

            if (!empty($attrCode)) {
                foreach ($cusRequest as $key => $val) {
                    $productCollection->addAttributeToFilter($key, ['in' => $val]);
                }
            }

            if (!empty($attrprice)) {
                if (!is_null($gteq)) {
                    $productCollection->addAttributeToFilter('price', ['gteq' => $gteq]);
                }
                if (!is_null($lteq)) {
                    $productCollection->addAttributeToFilter('price', ['lteq' => $lteq]);
                }
            }

            if (!empty($sortAttribute)) {
                $productCollection->addAttributeToSort($sortAttribute, $sortDirection);
            }

            $categoryCollection = $this->categoryCollectionFactory->create();
            $categoryCollection->addAttributeToSelect(['name','url_key']);
            $categoryCollection->addAttributeToFilter('is_active', ['eq' => 1]);
            $categoryCollection->addAttributeToFilter('name', ['like' => '%' . $searchkey . '%']);
            $categoryCollection->setCurPage($currentPage);

            $result['categories'] = $this->customApisHelper->getMainParentCategoriesNew($categoryCollection);
            
            foreach ($productCollection as $product) {
				
				// Calculate price with VAT
				$taxClassId = $product->getTaxClassId();
				$store      = $this->storeManager->getStore();
				$request = $this->taxCalculation->getRateRequest(null, null, null, $store);
				$taxRate = $this->taxCalculation->getRate($request->setProductClassId($taxClassId));
				
				$price = $product->getPrice();
				$price = round($price, 2);
				$basePrice = $price / 1.19;
				$priceWithTax = $basePrice + ($basePrice * ($taxRate / 100));
				
				$msrpPrice = $product->getMsrp();
				$msrpBasePrice = $msrpPrice / 1.19;
				$msrpPriceWithTax = $msrpBasePrice + ($msrpBasePrice * ($taxRate / 100));
				
                $stockItem = $this->stockRegistry->getStockItem($product->getId(), $product->getStore()->getWebsiteId());
                $manufacturerData = $this->getManufacturerData($product);
                $layerImageOptionId = $product->getLayerImages();
                $layerImagesPath = $this->getLayerImagesPath($layerImageOptionId);
                $isInWishlist = $this->isProductInWishlist($product, $customerId);
                $showButtons = $this->customApisHelper->getButtonStatus($product);
                $result['products'][] = [
                    'id' => $product->getId(),
                    'sku' => $product->getSku(),
                    'name' => $product->getName(),
                    'price' => (string)$priceWithTax,
                    'created_at' => $product->getCreatedAt(),
                    'extension_attributes' => [
                        'stock' => $stockItem->getQty(),
                        'is_in_stock' => $stockItem->getIsInStock(),
                    ],
                    'custom_attributes' => [
                        ['attribute_code' => 'delivery_time', 'value' => $product->getAttributeText('delivery_time')],
                        ['attribute_code' => 'msrp', 'value' => (string)$msrpPriceWithTax ?: ''],
                        ['attribute_code' => 'manufacturer', 'value' => $manufacturerData],
                        ['attribute_code' => 'layer_images', 'value' => $layerImagesPath],
                        ['attribute_code' => 'image', 'value' => $product->getData('image')],
                        ['attribute_code' => 'url_key', 'value' => $product->getData('url_key')],
                        ['attribute_code' => 'color', 'value' => $product->getAttributeText('color')],
                        ['attribute_code' => 'series', 'value' => $product->getAttributeText('series')],
                        ['attribute_code' => 'connection_size', 'value' => $product->getAttributeText('connection_size')],
                        ['attribute_code' => 'type2', 'value' => $product->getAttributeText('type2')],
                    ],
                    'is_in_wishlist' => $isInWishlist,
                    'average_rating' => $this->customApisHelper->getAverageRating($product->getId()),
                    'show_addtocart_button' => $showButtons,
                ];
            }

            $result['total_count_products'] = $productCollection->getSize();
            $result['total_count_categories'] = $categoryCollection->getSize();
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $result = [
                'error' => true,
                'message' => 'No products or categories found for the given query.',
            ];
        }

        return [$result];
    }

    private function getManufacturerData($product)
    {
        preg_match('/href="(.*?)"/', $product->getAttributeText('manufacturer'), $matches);
        $manufacturerImagePath = $this->getManufacturerImagePath($product);

        return [
            strip_tags($product->getAttributeText('manufacturer')),
            isset($matches[1]) ? str_replace('.html', '', $matches[1]) : '',
            $manufacturerImagePath,
        ];
    }

    private function isProductInWishlist($product,$customerId)
    {
        $isInWishlist = false;
        if (!empty($customerId)) {
            $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId, true);
            $wishlistItems = $wishlist->getItemCollection();
            foreach ($wishlistItems as $item) {
                if ($item->getProductId() == $product->getId()) {
                    $isInWishlist = true;
                    break;
                }
            }
        }

        return $isInWishlist;
    }

    private function getManufacturerImagePath($item)
    {
        return $this->imageHelper->getManufacturerImage($item) ?
            'attributeoption/image/' . $this->imageHelper->getManufacturerImage($item) : '';
    }

    private function getLayerImagesPath($optionId)
    {
        if ($this->imageHelper->getLayerImage($optionId)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getLayerImage($optionId);
            return $imagePath;
        }else{
            return '';
        }
    }

}
