<?php
namespace Xtwo\AttributesGroup\Block\Product\View;

use Exception;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\Eav\Api\AttributeSetRepositoryInterface;
use Magento\Eav\Api\AttributeGroupRepositoryInterface;
use Magento\Eav\Model\Attribute;
use Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Registry;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

use Magento\Framework\Filesystem;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Directory\WriteInterface;

class CustomAttributes extends Template
{
    /**
     * @var Product
     */
    protected $_product = null;

    /**
     * @var \Magento\Eav\Api\AttributeSetRepositoryInterface
     */
    private AttributeSetRepositoryInterface $attributeSetRepository;
    /**
     * @var \Magento\Eav\Api\AttributeGroupRepositoryInterface
     */
    private AttributeGroupRepositoryInterface $attributeGroupRepository;
    private SearchCriteriaBuilder $searchCriteriaBuilder;
    private FilterBuilder $filterBuilder;
    private LoggerInterface $logger;
    private Attribute $eavAttribute;
    private StoreManagerInterface $storeManager;
    private UrlInterface $urlInterface;
    private Registry $registry;
    private ?CollectionFactory $attributeCollectionFactory;
    /**
     * @var \Xtwo\Catalog\Block\Product\View\Magento\Catalog\Api\ProductRepositoryInterface
     */
    private ProductRepositoryInterface $productRepository;

    /**
     * Base constructor.
     *
     * @param \Magento\Framework\View\Element\Template\Context $context
     * @param \Magento\Eav\Api\AttributeSetRepositoryInterface $attributeSetRepository
     * @param \Magento\Eav\Api\AttributeGroupRepositoryInterface $attributeGroupRepository
     * @param \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder
     * @param \Magento\Framework\Api\FilterBuilder $filterBuilder
     * @param \Magento\Eav\Model\Attribute $eavAttribute
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Framework\UrlInterface $urlInterface
     * @param \Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory|null $attributeCollectionFactory
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param array $data
     */
    public function __construct(
        Context $context,
        AttributeSetRepositoryInterface $attributeSetRepository,
        AttributeGroupRepositoryInterface $attributeGroupRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        FilterBuilder $filterBuilder,
        Attribute $eavAttribute,
        StoreManagerInterface $storeManager,
        UrlInterface $urlInterface,
        CollectionFactory $attributeCollectionFactory,
        LoggerInterface $logger,
        Registry $registry,
        ProductRepositoryInterface $productRepository,
        Filesystem $filesystem,
        array $data = []
    ) {
        parent::__construct($context, $data);

        $this->attributeSetRepository = $attributeSetRepository;
        $this->attributeGroupRepository = $attributeGroupRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->filterBuilder = $filterBuilder;
        $this->logger = $logger;
        $this->eavAttribute = $eavAttribute;
        $this->storeManager = $storeManager;
        $this->urlInterface = $urlInterface;
        $this->registry = $registry;
        $this->attributeCollectionFactory = $attributeCollectionFactory;
        $this->productRepository = $productRepository;

        $this->mediaDirectory = $filesystem->getDirectoryWrite(DirectoryList::MEDIA);

        $this->setTabTitle();
    }

    /**
     * @param string $path
     *
     * @return bool
     */
    public function checkFilePath(string $path)
    {
        return $this->mediaDirectory->isFile($path);
    }

    /**
     * Set tab title
     *
     * @return void
     */
    public function setTabTitle()
    {
        $this->setTitle(__('Details'));
    }

    /**
     * @param \Magento\Catalog\Model\Product $product
     * @param false $backendValues
     *
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getExtendedGroupedAttributesForProduct($product, $backendValues = false)
    {
        $groups = [];

        $attributeSet = $this->attributeSetRepository->get($product->getAttributeSetId());

        $this->searchCriteriaBuilder->addFilters([
            $this->filterBuilder
                ->setField('main_table.attribute_set_id')
                ->setConditionType('eq')
                ->setValue($attributeSet->getId())
                ->create()
        ]);

        $this->searchCriteriaBuilder->setPageSize(1000);
        $searchCriteria = $this->searchCriteriaBuilder->create();

        $attributeGroups = $this->attributeGroupRepository->getList($searchCriteria);

        foreach ($attributeGroups->getItems() as $attributeGroup) {

            try {
                $groupLabel = $attributeGroup->getAttributeGroupName();
                $attributes = $this->getAttributesOfGroup($attributeGroup);

                foreach ($attributes as $attribute) {
                   if ($attribute->getIsVisibleOnFront() &&
                        $product->hasData($attribute->getAttributeCode()) &&
                        $product->getData($attribute->getAttributeCode()) != null) {
                        if (!array_key_exists($groupLabel, $groups)) {
                            $groups[$groupLabel] = [];
                        }

                        $frontend = $product->getResource()->getAttribute($attribute->getAttributeCode())
                            ->setStoreId($product->getStoreId())
                            ->getFrontend();

                        $productAttribute = $product->getResource()->getAttribute($attribute->getAttributeCode());

                        $value = $frontend->getValue($product);
                        if ($attribute->getFrontendInput() == 'multiselect') {
                            $optionIds = $product->getData($attribute->getAttributeCode());
                            $value = [];
                            foreach (explode(',', $optionIds) as $optionId) {
                                if ($backendValues) {
                                    $value[] = [
                                        'frontend' => $productAttribute->setStoreId(
                                            $product->getStoreId()
                                        )->getSource()->getOptionText($optionId),
                                        'backend' => $productAttribute->setStoreId(0)->getSource()->getOptionText(
                                            $optionId
                                        ),
                                    ];
                                } else {
                                    $value[] = $frontend->getOption($optionId);
                                }
                            }
                        }

                        $value = $attribute->getAttributeCode() == 'weight' ? number_format(
                                $value,
                                2,
                                ',',
                                '.'
                            ) . ' kg' : $value;
                        $groups[$groupLabel][$attribute->getAttributeCode()] = array(
                            'attributeCode' => $attribute->getAttributeCode(),
                            'value' => $value,
                            'attributeLabel' => $attribute->getStoreLabel($product->getStoreId()),
                            'position' => $attribute->getData('position'),
                        ); // may be array
                    }
                }


            } catch (LocalizedException $e) {
                $this->logger->error($e->getMessage());
            } catch (Exception $e) {
                $this->logger->error($e->getMessage());
            }
        }

        return $groups;

    }

    /**
     * @param $attributeGroup
     *
     * @return \Magento\Eav\Model\ResourceModel\Entity\Attribute\Collection|mixed
     */
    protected function getAttributesOfGroup($attributeGroup)
    {
        if (!isset($this->attributeGroups[$attributeGroup->getId()])) {

            $attributeCollection = $this->attributeCollectionFactory->create();

            $attributeCollection->setAttributeSetFilter($attributeGroup->getAttributeSetId());
            $attributeCollection->setAttributeGroupFilter($attributeGroup->getId());
            $attributeCollection->getSelect()
                ->joinLeft(
                    ["cea" => 'catalog_eav_attribute'],
                    "main_table.attribute_id = cea.attribute_id"
                );


            $this->attributeGroups[$attributeGroup->getId()] = $attributeCollection
                ->load();
        }

        return $this->attributeGroups[$attributeGroup->getId()];
    }

    /**
     * Returns a Product
     *
     * @return Product
     */
    public function getProduct()
    {
        if (!$this->_product) {

            if ($this->registry->registry('simple_product_id')) {
                $this->_product = $this->productRepository->getById($this->registry->registry('simple_product_id'));
            } else {
                $this->_product = $this->registry->registry('product');

                if ($this->_product->getTypeId() === 'configurable'
                    && $this->_product->getDefaultConfigurationId() !== null) { //@TODO
                    $this->_product = $this->productRepository->getById($this->_product->getDefaultConfigurationId());
                }
            }
        }
        return $this->_product;
    }

    /**
     * @param string $type
     *
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getBaseUrl($type = UrlInterface::URL_TYPE_DIRECT_LINK)
    {
        return $this->_storeManager->getStore()->getBaseUrl($type);
    }


}
