<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\AttributeOption\Repository;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Io\File;
use Magento\Store\Model\StoreManagerInterface;
use Xtwo\AttributeOption\Api\ImageRepositoryInterface;
use Xtwo\AttributeOption\Api\Data\ImageInterface;
use Xtwo\AttributeOption\Api\Data\ImageInterfaceFactory;
use Xtwo\AttributeOption\Api\Data\ImageSearchResultsInterface;
use Xtwo\AttributeOption\Api\Data\ImageSearchResultsInterfaceFactory;
use Xtwo\AttributeOption\Model\ImageFactory;
use Xtwo\AttributeOption\Model\ResourceModel\Image;
use Xtwo\AttributeOption\Model\ResourceModel\Image\CollectionFactory;

class ImageRepository implements ImageRepositoryInterface
{
    /**
     * @var \Xtwo\AttributeOption\Model\ImageFactory
     */
    private $imageFactory;
    /**
     * @var \Xtwo\AttributeOption\Model\ResourceModel\Image
     */
    private $imageResourceModel;
    /**
     * @var \Magento\Framework\Api\SearchResultsInterface
     */
    private $searchResults;
    /**
     * @var \Xtwo\AttributeOption\Model\ResourceModel\Image\CollectionFactory
     */
    private $imageCollectionFactory;
    /**
     * @var \Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface
     */
    private $collectionProcessor;
    /**
     * @var \Xtwo\AttributeOption\Api\Data\ImageSearchResultsInterfaceFactory
     */
    private $searchResultFactory;
    /**
     * @var \Xtwo\AttributeOption\Api\Data\ImageInterfaceFactory
     */
    private $imageDataFactory;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private StoreManagerInterface $storeManager;
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private File $file;
    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private DirectoryList $dir;

    /**
     * ImageRepository constructor.
     *
     * @param \Xtwo\AttributeOption\Model\ImageFactory $imageFactory
     * @param \Xtwo\AttributeOption\Api\Data\ImageInterfaceFactory $imageDataFactory
     * @param \Xtwo\AttributeOption\Model\ResourceModel\Image $imageResourceModel
     * @param \Magento\Framework\Api\SearchResultsInterface $searchResults
     * @param \Xtwo\AttributeOption\Api\Data\ImageSearchResultsInterfaceFactory $searchResultFactory
     * @param \Xtwo\AttributeOption\Model\ResourceModel\Image\CollectionFactory $imageCollectionFactory
     * @param \Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface $collectionProcessor
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Framework\Filesystem\Io\File $file
     * @param \Magento\Framework\Filesystem\DirectoryList $dir
     */
    public function __construct(
        ImageFactory $imageFactory,
        ImageInterfaceFactory $imageDataFactory,
        Image $imageResourceModel,
        SearchResultsInterface $searchResults,
        ImageSearchResultsInterfaceFactory $searchResultFactory,
        CollectionFactory $imageCollectionFactory,
        CollectionProcessorInterface $collectionProcessor,
        StoreManagerInterface $storeManager,
        File $file,
        DirectoryList $dir
    ) {
        $this->imageFactory = $imageFactory;
        $this->imageResourceModel = $imageResourceModel;
        $this->searchResults = $searchResults;
        $this->imageCollectionFactory = $imageCollectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultFactory = $searchResultFactory;
        $this->imageDataFactory = $imageDataFactory;
        $this->storeManager = $storeManager;
        $this->file = $file;
        $this->dir = $dir;
    }

    /**
     * @inheritdoc
     */
    public function save(ImageInterface $imageData): ImageInterface
    {
        $image = $this->initializeImageData($imageData->__toArray());
        $this->imageResourceModel->save($image);

        return $this->imageDataFactory->create()
            ->setId($image->getId())
            ->setFileName($image->getFileName())
            ->setOptionId($image->getOptionId())
            ->setAttributeId($image->getAttributeId())
            ->setStoreId($image->getStoreId());
    }

    /**
     * Merges data from DB and updates from request
     *
     * @param array $imageData
     *
     * @return \Xtwo\AttributeOption\Model\Image
     * @throws NoSuchEntityException
     */
    protected function initializeImageData(array $imageData)
    {
        if (!empty($imageData['entityid'])) {
            $image = $this->getById($imageData['id']);
        } else {
            $image = $this->imageFactory->create();
        }

        foreach ($imageData as $key => $value) {
            $image->setData($key, $value);
        }

        return $image;
    }

    /**
     * @inheritdoc
     */
    public function getById(int $id): ImageInterface
    {
        $image = $this->imageFactory->create();
        $image->load($id);

        $imageDataObject = $this->imageDataFactory->create()
            ->setId($image->getId())
            ->setFileName($image->getFileName())
            ->setOptionId($image->getOptionId())
            ->setAttributeId($image->getAttributeId())
            ->setStoreId($image->getStoreId());

        return $imageDataObject;
    }

    /**
     * @inheritdoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria): ImageSearchResultsInterface
    {
        /** @var \Xtwo\AttributeOption\Model\ResourceModel\Image\Collection $collection */
        $collection = $this->imageCollectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        /** @var  $searchResults */
        $searchResults = $this->searchResultFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);

        $images = [];

        foreach ($collection as $image) {
            $imageDataObject = $this->imageDataFactory->create()
                ->setId($image->getEntityId())
                ->setFileName($image->getFileName())
                ->setOptionId($image->getOptionId())
                ->setAttributeId($image->getAttributeId())
                ->setStoreId($image->getStoreId());

            $images[] = $imageDataObject;
        }

        $searchResults->setTotalCount($collection->getSize());

        return $searchResults->setItems($images);
    }

    /**
     * @inheritdoc
     */
    public function deleteById(int $id): bool
    {
        $model = $this->imageFactory->create()->load($id);

        if (!$model->getId()) {
            throw new NoSuchEntityException();
        }

        $path = $this->dir->getPath('media') . '/' . \Xtwo\AttributeOption\Model\Image::PATH_TO_LOGO . $id;
        $this->file->rmdir($path, true);

        $this->imageResourceModel->delete($model);

        return true;
    }
}
