<?php

namespace Xtwo\AttributeOption\Console\Command;

use Magento\Catalog\Model\Product;
use Magento\Eav\Model\Config;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Exception;
use Symfony\Component\Console\Style\SymfonyStyle;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Filesystem\Io\File as IoFile;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\FileSystemException;
use Xtwo\AttributeOption\Api\Data\ImageInterfaceFactory;
use Xtwo\AttributeOption\Model\ImageUploader;
use Xtwo\AttributeOption\Repository\ImageRepository;
use Xtwo\AttributeOption\Model\Image;

class ImportAttributeImageCommand extends Command
{
    const FILENAME = 'filename';

    const ATTRIBUTE_CODE = 'attribute_code';

    /**
     * @var \Xtwo\AttributeOption\Api\Data\ImageInterfaceFactory
     */
    private ImageInterfaceFactory $imageDataFactory;
    /**
     * @var \Xtwo\AttributeOption\Model\ImageUploader
     */
    private ImageUploader $imageUploader;
    /**
     * @var \Magento\Framework\App\Filesystem\DirectoryList
     */
    private DirectoryList $directoryList;
    /**
     * @var \Magento\Framework\File\Csv
     */
    private Csv $csv;
    /**
     * @var \Magento\Framework\Filesystem\Driver\File
     */
    private File $file;
    /**
     * @var \Xtwo\AttributeOption\Repository\ImageRepository
     */
    private ImageRepository $imageRepository;
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private IoFile $ioFile;
    /**
     * @var \Magento\Eav\Model\Config
     */
    private Config $eavConfig;

    /**
     * ImportAttributeImageCommand constructor.
     *
     * @param \Magento\Framework\App\Filesystem\DirectoryList $directoryList
     * @param \Magento\Framework\File\Csv $csv
     * @param \Magento\Framework\Filesystem\Driver\File $file
     * @param \Xtwo\AttributeOption\Api\Data\ImageInterfaceFactory $imageDataFactory
     * @param \Xtwo\AttributeOption\Model\ImageUploader $imageUploader
     * @param \Xtwo\AttributeOption\Repository\ImageRepository $imageRepository
     * @param \Magento\Framework\Filesystem\Io\File $ioFile
     * @param \Magento\Eav\Model\Config $eavConfig
     */
    public function __construct(
        DirectoryList $directoryList,
        Csv $csv,
        File $file,
        ImageInterfaceFactory $imageDataFactory,
        ImageUploader $imageUploader,
        ImageRepository $imageRepository,
        IoFile $ioFile,
        Config $eavConfig
    )
    {
        parent::__construct();

        $this->imageDataFactory = $imageDataFactory;
        $this->imageUploader = $imageUploader;
        $this->directoryList = $directoryList;
        $this->csv = $csv;
        $this->file = $file;
        $this->imageRepository = $imageRepository;
        $this->ioFile = $ioFile;
        $this->eavConfig = $eavConfig;
    }


    /**
     * @param \Symfony\Component\Console\Input\InputInterface $input
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     *
     * @return int|null
     * @throws \Exception
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        try {
            if (!$input->hasOption(self::FILENAME)) {
                throw new Exception(__('No file name is provided'));
            }
            if (!$input->hasOption(self::ATTRIBUTE_CODE)) {
                throw new Exception(__('No attribute code is provided'));
            }

            $rootDirectory = $this->directoryList->getRoot();
            $csvFile = $rootDirectory . "/" . $input->getOption(self::FILENAME);
            $attribute = $this->eavConfig->getAttribute(
                Product::ENTITY,
                $input->getOption(self::ATTRIBUTE_CODE)
            );

            if ($this->file->isExists($csvFile)) {
                $this->csv->setDelimiter(",");
                $data = $this->csv->getData($csvFile);

                if (!empty($data)) {

                    foreach (array_slice($data, 1) as $key => $value) {
                        $image = $this->imageDataFactory->create();

                        $image->setFileName(trim($value[3]));
                        $image->setOptionId(trim($value[1]));
                        $image->setAttributeId($attribute->getId());
                        $image->setStoreId($value[2]);
                        $image = $this->imageRepository->save($image);

                        $id = $image->getId();
                        $path = $this->directoryList->getPath('media') . '/' . Image::PATH_TO_LOGO . $id;
                        if (!file_exists($path)) {
                            $this->ioFile->mkdir($path);
                        }

                        $this->imageUploader->moveFileFromTmpToPath(trim($value[3]), '/' . $id . '/');
                    }

                }
            } else {
                $this->_logger->info('Csv file not exist');
                throw new Exception(__('Csv file not exist'));
            }

        } catch (Exception $exception) {
            $io->error("");
            throw new Exception($exception->getMessage());

            return 0;
        }

        $io->success('The attribute image option were imported');

        return 0;
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $options = [
            new InputOption(
                self::FILENAME,
                null,
                InputOption::VALUE_REQUIRED,
                "File name"
            ),
            new InputOption(
                self::ATTRIBUTE_CODE,
                null,
                InputOption::VALUE_REQUIRED,
                "Attribute code"
            ),
        ];

        $this->setName('xtwo:attribute-option:import-attribute-image');
        $this->setDescription('Import attribute image service.');
        $this->setDefinition($options);

        $helpText = <<<TEXT
 Usage: xtwo:attribute-option:import-attribute-image

TEXT;
        $this->addUsage($helpText);

        parent::configure();
    }

}
